/***********************************************************************************************
 * @file    bc3603spi.c
 * @version $Rev:: 4837         $
 * @date    $Date:: 2022-09-02 #$
 * @brief   This file provides all the RF firmware functions.
 ***********************************************************************************************
 * @attention
 *
 * Firmware Disclaimer Information
 *
 * 1. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, which is supplied by Holtek Semiconductor Inc., (hereinafter referred to as "HOLTEK") is the
 *    proprietary and confidential intellectual property of HOLTEK, and is protected by copyright law and
 *    other intellectual property laws.
 *
 * 2. The customer hereby acknowledges and agrees that the program technical documentation, including the
 *    code, is confidential information belonging to HOLTEK, and must not be disclosed to any third parties
 *    other than HOLTEK and the customer.
 *
 * 3. The program technical documentation, including the code, is provided "as is" and for customer reference
 *    only. After delivery by HOLTEK, the customer shall use the program technical documentation, including
 *    the code, at their own risk. HOLTEK disclaims any expressed, implied or statutory warranties, including
 *    the warranties of merchantability, satisfactory quality and fitness for a particular purpose.
 *
 * <h2><center>Copyright (C) Holtek Semiconductor Inc. All rights reserved</center></h2>
 ***********************************************************************************************/

/* Includes -----------------------------------------------------------------------------------*/
#include	"ht32.h"
#include	"bc3603.h"
#include "bc3603spi.h"

/* Private constants --------------------------------------------------------------------------*/
/** @defgroup RF_Private_Define RF private definitions
  * @{
  */

/**
  * @}
  */

/* Global functions ---------------------------------------------------------------------------*/
/** @defgroup RF_Exported_Functions RF exported functions
  * @{
  */
#if	(RFINF_SPI == RFINF_FW_SPI)
/***********************************************************************************************
 * @brief Firmware SPI serial data output.
 * @param data -> output data
 * @retval None
 ***********************************************************************************************/
static void BC3603_SPISerialOutput(u8 data)
{
   u8 x;
	
   for(x=0;x<8;x++)
   {
      SPISCK_RESET();            
     	(data & 0x80) ? SPIMOSI_SET():SPIMOSI_RESET();
      SPISCK_SET();
      data <<= 1;
   }
	__NOP();			
   __NOP();   
   SPISCK_RESET();
}
/***********************************************************************************************
 * @brief Firmware SPI serial data input.
 * @param None
 * @retval SPI data 
 ***********************************************************************************************/
static u8 BC3603_SPISerialInput(void)
{
	u8 x,d;
   d = 0;
	
   for(x=0;x<8;x++)
   {
		d <<= 1;
      SPISCK_SET();
		if(SPIMISO_IN()) d |= 0x01;
      SPISCK_RESET();
   }
   return d;
}
#endif
/***********************************************************************************************
 * @brief Configure EXTI to RF IRQ.
 * @param None
 * @retval None
 ***********************************************************************************************/
void BC3603_IRQExtiConfigure(void)
{
	EXTI_InitTypeDef	EXTI_InitStruct;
	
	AFIO_EXTISourceConfig(RF_IRQ_EXIT_CH,RF_IRQ_AFIO_ESS);
	EXTI_InitStruct.EXTI_Channel = RF_IRQ_EXIT_CH;
	EXTI_InitStruct.EXTI_Debounce = EXTI_DEBOUNCE_DISABLE;
	EXTI_InitStruct.EXTI_DebounceCnt = 0;
	EXTI_InitStruct.EXTI_IntType = (_IRQ_POLARITY_ == 1) ? EXTI_POSITIVE_EDGE:EXTI_NEGATIVE_EDGE;
	EXTI_Init(&EXTI_InitStruct);	
}
/***********************************************************************************************
 * @brief Configure EXTI to RF IRQ.
 * @param None
 * @retval None
 ***********************************************************************************************/
void BC3603_IRQExtiControl(ControlStatus NewState)
{
	EXTI_IntConfig(RF_IRQ_EXIT_CH,NewState);
	NVIC_ClearPendingIRQ(RF_IRQn);	
	if(NewState == ENABLE)	NVIC_EnableIRQ(RF_IRQn);
	else	                  NVIC_DisableIRQ(RF_IRQn);	
}
/***********************************************************************************************
 * @brief Get IRQ status.
 * @param None
 * @retval None
 ***********************************************************************************************/
FlagStatus BC3603_IRQLineStatus(void)
{	
#if (_IRQ_POLARITY_ == 1)
	if(RFIRQ_IN()) 
		return(SET);
	else				
		return(RESET);
#else	
	if(!RFIRQ_IN())	
		return(SET);
	else					
		return(RESET);	
#endif	
}
/***********************************************************************************************
 * @brief MCU & RF interface configure.
 * @param None
 * @retval None
 ***********************************************************************************************/
void BC3603_InterfaceConfigure(void)
{
	CKCU_PeripClockConfig_TypeDef CKCUClock = {{0}};
#if	(RFINF_SPI == RFINF_HW_SPI)	
	SPI_InitTypeDef  SPI_InitStructure;
#endif	

	/* Enable peripheral clock of PortB */
	CKCUClock.Bit.PB = ENABLE;
#if	(RFINF_SPI == RFINF_HW_SPI)
	/* Enable peripheral clock of SPI0 */
	CKCUClock.Bit.SPI0 = ENABLE;	
#endif	
	CKCU_PeripClockConfig(CKCUClock, ENABLE);
	
	/* Config IRQ AFIO mode */
	AFIO_GPxConfig(RF_IRQ_AFIO_PORT, RF_IRQ, RF_IRQ_AFIO_FUN);
	/* Config IRQ for input mode/pull-high/input enable */		
	GPIO_DirectionConfig(RF_IRQ_PORT, RF_IRQ, GPIO_DIR_IN);
	GPIO_PullResistorConfig(RF_IRQ_PORT, RF_IRQ, GPIO_PR_UP);
   GPIO_InputConfig(RF_IRQ_PORT, RF_IRQ, ENABLE); 

#if	(RFINF_SPI == RFINF_FW_SPI)
	/* Config SCK & MOSI & MISO AFIO mode */
	AFIO_GPxConfig(SPI_CSN_AFIO_PORT, SPI_CSN, SPI_CSN_AFIO_FUN);	
	AFIO_GPxConfig(SPI_SCK_AFIO_PORT, SPI_SCK, SPI_SCK_AFIO_FUN);
	AFIO_GPxConfig(SPI_MOSI_AFIO_PORT, SPI_MOSI, SPI_MOSI_AFIO_FUN);
	AFIO_GPxConfig(SPI_MISO_AFIO_PORT, SPI_MISO, SPI_MISO_AFIO_FUN);
	/* Config SCK & MOSI for output mode */	
	GPIO_DirectionConfig(SPI_CSN_PORT, SPI_CSN, GPIO_DIR_OUT);	
	GPIO_DirectionConfig(SPI_SCK_PORT, SPI_SCK, GPIO_DIR_OUT);
	GPIO_DirectionConfig(SPI_MOSI_PORT, SPI_MOSI, GPIO_DIR_OUT);
	/* Config MISO for input mode/pull-high/input enable */	
	GPIO_DirectionConfig(SPI_MISO_PORT, SPI_MISO, GPIO_DIR_IN);
	GPIO_PullResistorConfig(SPI_MISO_PORT, SPI_MISO, GPIO_PR_UP);
   GPIO_InputConfig(SPI_MISO_PORT, SPI_MISO, ENABLE); 
	SPI_INACTIVE();	
	SPISCK_RESET();
	SPIMOSI_RESET();
#else
   /* SPI Configuration */	
	SPI_InitStructure.SPI_Mode = SPI_MASTER;
	SPI_InitStructure.SPI_FIFO = SPI_FIFO_DISABLE;
	SPI_InitStructure.SPI_DataLength = SPI_DATALENGTH_8;
	SPI_InitStructure.SPI_SELMode = SPI_SEL_SOFTWARE;
	SPI_InitStructure.SPI_SELPolarity = SPI_SELPOLARITY_LOW;
	SPI_InitStructure.SPI_FirstBit = SPI_FIRSTBIT_MSB;
	SPI_InitStructure.SPI_CPOL = SPI_CPOL_LOW;
	SPI_InitStructure.SPI_CPHA = SPI_CPHA_FIRST;
   /*-- SPIx FIFO Control Register Configuration --*/	
	SPI_InitStructure.SPI_RxFIFOTriggerLevel = 0;
	SPI_InitStructure.SPI_TxFIFOTriggerLevel = 0;
   /*-- SPIx Clock Prescaler Register Configuration -*/	
	if((CKCU_GetPeripFrequency(RFSPI_PCLK) / 2) >= RFSPI_SPEED)
		SPI_InitStructure.SPI_ClockPrescaler = CKCU_GetPeripFrequency(RFSPI_PCLK) / RFSPI_SPEED;
	else
		SPI_InitStructure.SPI_ClockPrescaler = 2;				/* fSCK=fPCLK/2 */
	SPI_Init(RFSPI, &SPI_InitStructure);	
   /* Enable the SEL output for the specified SPI peripheral.*/	
	SPI_SELOutputCmd(RFSPI, ENABLE);
   /* Disable the specified SPI interrupt */	
	SPI_IntConfig(RFSPI, SPI_INT_ALL, DISABLE);
   /* Enable the selected SPI peripheral */	
	SPI_Cmd(RFSPI, ENABLE);
	/* Config SCK & MOSI & MISO AFIO mode */
	AFIO_GPxConfig(SPI_CSN_AFIO_PORT, SPI_CSN, SPI_CSN_AFIO_FUN);	
	AFIO_GPxConfig(SPI_SCK_AFIO_PORT, SPI_SCK, SPI_SCK_AFIO_FUN);
	AFIO_GPxConfig(SPI_MOSI_AFIO_PORT, SPI_MOSI, SPI_MOSI_AFIO_FUN);
	AFIO_GPxConfig(SPI_MISO_AFIO_PORT, SPI_MISO, SPI_MISO_AFIO_FUN);
	GPIO_PullResistorConfig(SPI_MISO_PORT, SPI_MISO, GPIO_PR_UP);	
	SPI_INACTIVE();		
#endif
}
/***********************************************************************************************
 * @brief Transmission 1-byte strobe command.
 * @param cmd: 1-byte strobe command
 * @retval None
 ***********************************************************************************************/
void BC3603_StrobeCommand(u8 Cmd)
{
#if	(RFINF_SPI == RFINF_FW_SPI)         
   SPI_ACTIVE();                                 		/* enable CS */
   BC3603_SPISerialOutput(Cmd);                       		/* transmission command */
   SPI_INACTIVE();                                		/* disable CS */
#else
	SPI_SetDataLength(RFSPI,SPI_DATALENGTH_8);			/* data length=8bits */   
   SPI_ACTIVE();     											/* enable CS */	
   RFSPI->DR = Cmd;												/* transmission command */
   while(!(RFSPI->SR & SPI_FLAG_TXE)) __NOP();        /* wait transmission finish */ 
   __NOP();	
   SPI_INACTIVE();												/* disable CS */					
   __NOP();
   Cmd = RFSPI->DR;   											/* clear SPI RX buffer */   	
#endif      
}
/***********************************************************************************************
 * @brief read data from RF register.
 * @param addr: register address.
 * @retval register data(u8)
 ***********************************************************************************************/
u8 BC3603_ReadRegister(u8 Addr)
{
   u8 value;
   
   Addr &= REGSADDR_MASK;										/* mask register address */
   Addr |= READ_REGS_CMD;                             /* set read register command */
#if	(RFINF_SPI == RFINF_FW_SPI)
   SPI_ACTIVE();                                 		/* enable CS */
   BC3603_SPISerialOutput(Addr);                          /* transmission command */
   value = BC3603_SPISerialInput();                       /* transmission data */
   SPI_INACTIVE();                               		/* disable CS */
#else  
	SPI_SetDataLength(RFSPI,SPI_DATALENGTH_16);			/* data length=16bits */   
   SPI_ACTIVE();     											/* enable CS */	
   RFSPI->DR = (Addr << 8) + 0x00FF;						/* transmission command & data */
   while(!(RFSPI->SR & SPI_FLAG_TXE)) __NOP();			/* wait transmission finish */
   SPI_INACTIVE();                                    /* disable CS */
   while(!(RFSPI->SR & SPI_FLAG_RXBNE)) __NOP();      /* wait Rx buffer ready */
   value = (u8)(RFSPI->DR & 0xFF);							/* read transmission data */
#endif      
   return (value);
}
/***********************************************************************************************
 * @brief write data to RF register.
 * @param addr: register address.
 * @param data: write data.
 * @retval None
 ***********************************************************************************************/
void BC3603_WriteRegister(u8 Addr,u8 Data)
{   
   Addr &= REGSADDR_MASK;										/* mask register address */
   Addr |= WRITE_REGS_CMD;                            /* set write register command */
#if	(RFINF_SPI == RFINF_FW_SPI)            
   SPI_ACTIVE();			                             	/* enable CS */
   BC3603_SPISerialOutput(Addr);                          /* transmission command */
   BC3603_SPISerialOutput(Data);                          /* transmission data */
   SPI_INACTIVE();												/* disable CS */
#else   
	SPI_SetDataLength(RFSPI,SPI_DATALENGTH_16);			/* data length=16bits */   
   SPI_ACTIVE();     											/* enable CS */	
   RFSPI->DR = (Addr << 8) + Data;							/* transmission command & data */
   while(!(RFSPI->SR & SPI_FLAG_TXE)) __NOP();			/* wait transmission finish */
   SPI_INACTIVE();                                    /* disable CS */
   while(!(RFSPI->SR & SPI_FLAG_RXBNE)) __NOP();      /* wait Rx buffer ready */
   Data = RFSPI->DR;												/* clear SPI RX buffer */	
#endif   
}
/***********************************************************************************************
 * @brief read multiple data from RF register.
 * @param cmd: strobe command.
 * @param pbuf: data buffer.
 * @param length: read length.
 * @retval None
 ***********************************************************************************************/
void BC3603_ReadMultipleRegister(u8 Cmd,u8 *DstAddr,u16 Length)
{
#if	(RFINF_SPI == RFINF_FW_SPI)            
   SPI_ACTIVE();                                     /* enable CS */
   BC3603_SPISerialOutput(Cmd);                          /* transmission command */
   while(Length --)
   {
      *DstAddr++ = BC3603_SPISerialInput();             	/* transmission data */
   }
   SPI_INACTIVE();
#else   
	SPI_SetDataLength(RFSPI,SPI_DATALENGTH_8);			/* data length=8bits */   
   SPI_ACTIVE();     											/* enable CS */	
   RFSPI->DR = Cmd;												/* transmission command */
	while(!(RFSPI->SR & SPI_FLAG_TXBE)) __NOP();	      /* wait buffer empty */
	while(!(RFSPI->SR & SPI_FLAG_RXBNE)) __NOP();		/* wait rx buffer ready */
   Cmd = RFSPI->DR;             								/* clear SPI RX buffer */
   while(Length--)
   {
      RFSPI->DR = 0xFF;
		while(!(RFSPI->SR & SPI_FLAG_TXBE)) __NOP();		/* wait buffer empty */
		if(RFSPI->SR & SPI_FLAG_RXBNE) *DstAddr++ = RFSPI->DR;
//      while(!(RFSPI->SR & SPI_FLAG_RXBNE)) __NOP();	/* wait rx buffer ready */
//      *DstAddr++ = RFSPI->DR;                         /* read transmission data */
   }
	while(!(RFSPI->SR & SPI_FLAG_TXE)) __NOP();			/* wait transmission finish */
   SPI_INACTIVE();                                    /* disable CS */
   while(!(RFSPI->SR & SPI_FLAG_RXBNE)) __NOP();		/* wait rx buffer ready */	
	*DstAddr++ = RFSPI->DR;
#endif      
}
/***********************************************************************************************
 * @brief write multiple data to RF register.
 * @param cmd: strobe command.
 * @param pbuf: data buffer.
 * @param length: read length.
 * @retval None
 ***********************************************************************************************/
void BC3603_WriteMultipleRegister(u8 Cmd,u8 *ScrAddr,u16 Length)
{
#if	(RFINF_SPI == RFINF_FW_SPI)            
   SPI_ACTIVE();													/* enable CS */
   BC3603_SPISerialOutput(Cmd);                           /* transmission command */
   while(Length --)
   {
      BC3603_SPISerialOutput(*ScrAddr++);                 /* transmission data */
   }
   SPI_INACTIVE();                                    /* disable CS */
#else
	SPI_SetDataLength(RFSPI,SPI_DATALENGTH_8);			/* data length=8bits */   
   SPI_ACTIVE();     											/* enable CS */	
   RFSPI->DR = Cmd;												/* transmission command */
   while(Length --)
   {
      while(!(RFSPI->SR & SPI_FLAG_TXBE)) __NOP();	   /* wait buffer empty */
      RFSPI->DR = *ScrAddr++;                 			/* transmission data */
   }
   Cmd = RFSPI->DR;                      					/* clear SPI RX buffer */
   while(!(RFSPI->SR & SPI_FLAG_TXE)) __NOP();			/* wait transmission finish */
   SPI_INACTIVE();                                    /* disable CS */
   while(!(RFSPI->SR & SPI_FLAG_RXBNE)) __NOP();
   while(RFSPI->SR & SPI_FLAG_RXBNE) Cmd = RFSPI->DR;	/* clear SPI RX buffer */
#endif   
}

/**
  * @}
  */
